import { defineStore } from 'pinia'
import { authService } from '@/services/auth.service'
import type { AccountInfo } from '@azure/msal-browser'

interface UserState {
  account: AccountInfo | null
  loading: boolean
  error: string | null
  lastAction: string | null
}

export const useUserStore = defineStore('user', {
  state: (): UserState => ({
    account: null,
    loading: false,
    error: null,
    lastAction: null
  }),
  
  getters: {
    isLoggedIn: state => !!state.account,
    allAccounts: () => authService.getAllAccounts(),
    currentAccount: state => state.account,
    hasError: state => !!state.error
  },
  
  actions: {
    async initialize() {
      try {
        this.loading = true
        this.error = null
        this.lastAction = 'initialize'
        const account = authService.getCurrentAccount()
        if (account) {
          this.account = account
        }
      } catch (error) {
        this.handleError(error, 'Initialize failed')
      } finally {
        this.loading = false
      }
    },

    async login() {
      try {
        this.loading = true
        this.error = null
        this.lastAction = 'login'
        await authService.login()
        this.account = authService.getCurrentAccount()
      } catch (error) {
        this.handleError(error, 'Login failed')
        throw error
      } finally {
        this.loading = false
      }
    },

    async logout() {
      try {
        this.loading = true
        this.error = null
        this.lastAction = 'logout'
        await authService.logout()
        this.account = null
      } catch (error) {
        this.handleError(error, 'Logout failed')
        throw error
      } finally {
        this.loading = false
      }
    },

    async switchAccount() {
      try {
        this.loading = true
        this.error = null
        this.lastAction = 'switchAccount'
        const response = await authService.switchAccount()
        this.account = response.account
      } catch (error) {
        this.handleError(error, 'Account switch failed')
        throw error
      } finally {
        this.loading = false
      }
    },

    handleError(error: unknown, defaultMessage: string) {
      this.error = error instanceof Error ? error.message : defaultMessage
      console.error(`${this.lastAction} error:`, error)
    },

    clearError() {
      this.error = null
    }
  }
})