import { 
  PublicClientApplication, 
  EventType,
  InteractionRequiredAuthError,
  type AccountInfo, 
  type AuthenticationResult
} from '@azure/msal-browser'
import { msalInstance, loginRequest, tokenRequest, usePopupLogin, usePopupLogout } from '@/config/auth'

export class AuthService {
  private static instance: AuthService
  private currentAccount: AccountInfo | null = null

  private constructor() {
    // 确保 MSAL 实例初始化
    console.log('MSAL Instance:', msalInstance)

    // 监听账户变更
    msalInstance.addEventCallback((event) => {
      if (event.eventType === EventType.ACCOUNT_ADDED || 
          event.eventType === EventType.ACCOUNT_REMOVED || 
          event.eventType === EventType.LOGIN_SUCCESS ||
          event.eventType === EventType.SSO_SILENT_SUCCESS) {
        const accounts = msalInstance.getAllAccounts()
        if (accounts.length > 0) {
          this.currentAccount = accounts[0]
          msalInstance.setActiveAccount(accounts[0])
        } else {
          this.currentAccount = null
          msalInstance.setActiveAccount(null)
        }
      }
    })

    // 初始化时设置活跃账号
    const accounts = msalInstance.getAllAccounts()
    if (accounts.length > 0) {
      this.currentAccount = accounts[0]
      msalInstance.setActiveAccount(accounts[0])
    }
  }

  public static getInstance(): AuthService {
    if (!AuthService.instance) {
      AuthService.instance = new AuthService()
    }
    return AuthService.instance
  }

  async login(): Promise<void> {
    try {
      if (usePopupLogin) {
        const response = await msalInstance.loginPopup(loginRequest)
        this.currentAccount = response.account
        msalInstance.setActiveAccount(response.account)
      } else {
        await msalInstance.loginRedirect(loginRequest)
        // 重定向登录不返回 AuthenticationResult
        // 登录成功后的逻辑在重定向回调中处理
      }
    } catch (error) {
      console.error('Login failed:', error)
      throw error
    }
  }

  async switchAccount(): Promise<AuthenticationResult> {
    try {
      const response = await msalInstance.loginPopup({
        ...loginRequest,
        prompt: 'select_account'
      })
      this.currentAccount = response.account
      msalInstance.setActiveAccount(response.account)
      return response
    } catch (error) {
      console.error('Account switch failed:', error)
      throw error
    }
  }

  async logout(): Promise<void> {
    try {
      if (usePopupLogout) {
        await msalInstance.logoutPopup({
          account: this.currentAccount ?? undefined
        })
      } else {
        await msalInstance.logoutRedirect({
          account: this.currentAccount ?? undefined
        })
      }
      this.currentAccount = null
    } catch (error) {
      console.error('Logout failed:', error)
      throw error
    }
  }

  async getToken(): Promise<string> {
    try {
      const account = msalInstance.getActiveAccount()
      if (!account) {
        throw new Error('No active account!')
      }

      const response = await msalInstance.acquireTokenSilent({
        ...tokenRequest,
        account
      })
      return response.accessToken
    } catch (error) {
      console.error('Token acquisition failed:', error)
      throw error
    }
  }

  isAuthenticated(): boolean {
    return !!this.currentAccount
  }

  getCurrentAccount(): AccountInfo | null {
    return msalInstance.getActiveAccount()
  }

  getAllAccounts(): AccountInfo[] {
    return msalInstance.getAllAccounts()
  }
}

export const authService = AuthService.getInstance() 