
Page({
  /**
   * 页面的初始数据
   */
  data: {
    selectedTab: 'home', // 默认选中的标签
    location: '', // 位置信息
    nearestLocation: '', // 最近打卡点
    nearestLocationId: '', // 最近打卡点的ID
    canCheckIn: false, // 是否可以打卡
    address: '',
    userId: '', // 用户唯一标志
    userRank: '?'
  },
  showb1() {
    this.setData({
      styleA: 'transform:rotateY(180deg)',
      styleB: 'transform:rotateY(0deg)'
    });
  },
  showb2() {
    this.setData({
      styleA: 'transform:rotateY(0deg)',
      styleB: 'transform:rotateY(-180deg)'
    });
  },
  navigateToHome() {
    this.setData({
      selectedTab: 'home'
    });
    wx.navigateBack({
      delta: 1 // 返回上一个页面
    });
  },
  navigateToRank() {
    this.setData({
      selectedTab: 'rank'
    });
    wx.navigateTo({
      url: '/pages/rank/rank'
    });
  },
  navigateToCard() {
    this.setData({
      selectedTab: 'card'
    });
    wx.navigateTo({
      url: '/pages/subPack/roateY180/index'
    });
  },
  onLoad() {
    this.showb2();
    setTimeout(() => {
      this.showb1();
    }, 2000); // 延迟1秒调用showb2
    // 每隔20秒刷新一次位置
    this.updateLocation();
    // setInterval(this.updateLocation.bind(this), 20000);
    // 等待云实例初始化完成调用登录方法
    this.waitForSpecificValue().then(() => {
      this.userLogin();
  });
  },
  /**
   * 模拟获取特定值的方法
   * 返回一个Promise，特定值有效时resolve
   */
  waitForSpecificValue() {
    console.log("info:正在等待云实例初始化完成")
    return new Promise((resolve) => {
      const checkValueInterval = setInterval(() => {
        if (getApp().globalData.cloudInstance) {
          console.log("info:云实例已完成初始化")
          clearInterval(checkValueInterval);
          resolve();
        }
      }, 100); // 检查间隔（毫秒）
    });
  },
  updateLocation() {
    wx.getLocation({
      type: 'wgs84',
      success: (res) => {
        const latitude = res.latitude;
        const longitude = res.longitude;
        this.setData({
          location: `纬度: ${latitude}, 经度: ${longitude}`
        });
        this.checkProximity(latitude, longitude);
        this.getAddress(latitude, longitude);
      },
      fail: (err) => {
        console.error('获取位置信息失败', err);
        this.setData({
          location: '',
          canCheckIn: false
        });
      }
    });
  },
  checkProximity(latitude, longitude) {
    const app = getApp();
    const db = app.globalData.cloudInstance.database();
    const _ = db.command;
    const range = 4; // 2公里范围
    // 计算经纬度范围
    const latRange = range / 111; // 1度纬度约等于111公里
    const lonRange = range / (111 * Math.cos(latitude * Math.PI / 180)); // 1度经度约等于111公里 * cos(纬度)
    console.log("info:维度及经度",latitude, longitude)
    console.log("info:维度范围",latitude - latRange, latitude + latRange)
    console.log("info:经度范围",longitude - lonRange, longitude + lonRange)
    db.collection('check_in_location').where({
      latitude: _.gte(latitude - latRange).and(_.lte(latitude + latRange)),
      longitude: _.gte(longitude - lonRange).and(_.lte(longitude + lonRange))
    }).get({
      success: (res) => {
        console.log('info:check_in_location筛选附近打卡点结果', res)
        if (res.data.length > 0) {
          let nearestLocation = res.data[0];
          let minDistance = this.calculateDistance(latitude, longitude, nearestLocation.latitude, nearestLocation.longitude);
          res.data.forEach(location => {
            const distance = this.calculateDistance(latitude, longitude, location.latitude, location.longitude);
            if (distance < minDistance) {
              minDistance = distance;
              nearestLocation = location;
            }
          });
          // 刷新登录及排名信息
          this.userLogin()
          this.setData({
            nearestLocation: nearestLocation.location_name,
            nearestLocationId: nearestLocation.location_id,
            canCheckIn: minDistance <= nearestLocation.checkin_range
          });
        } else {
          this.setData({
            nearestLocation: '',
            canCheckIn: false
          });
        }
      },
      fail: (err) => {
        console.error('获取打卡地点失败', err);
      }
    });
  },
  calculateDistance(lat1, lon1, lat2, lon2) {
    const R = 6371e3; // 地球半径，单位为米
    const φ1 = lat1 * Math.PI / 180;
    const φ2 = lat2 * Math.PI / 180;
    const Δφ = (lat2 - lat1) * Math.PI / 180;
    const Δλ = (lon2 - lon1) * Math.PI / 180;
    const a = Math.sin(Δφ / 2) * Math.sin(Δφ / 2) +
      Math.cos(φ1) * Math.cos(φ2) *
      Math.sin(Δλ / 2) * Math.sin(Δλ / 2);
    const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
    const distance = R * c; // 单位为米
    return distance;
  },
  async handleCheckIn() {
    const that = this;
    // 有用户id执行打卡，无用户id先登录，后执行打卡逻辑
    if (that.data.userId) {
      that.getLocationAndCheckIn();
    } else {
      this.userLogin();
      that.getLocationAndCheckIn();
    }
  },
  // 打卡并写入集合
  getLocationAndCheckIn() {
    if (!this.data.canCheckIn || this.data.userRank != '?') {
      wx.showToast({
        title: '无法重复打卡,请稍后再试！',
        icon: 'none',
        duration: 2000
      });
      return;
    }
    wx.getLocation({
      type: 'wgs84',
      success: (res) => {
        const latitude = res.latitude;
        const longitude = res.longitude;
        this.setData({
          location: `纬度: ${latitude}, 经度: ${longitude}`
        });
        this.getAddress(latitude, longitude);
        // 写入 check_in_record 集合
        const app = getApp();
        const db = app.globalData.cloudInstance.database();
        db.collection('check_in_record').add({
          data: {
            location_id: this.data.nearestLocationId,
            latitude: latitude,
            longitude: longitude,
            address: this.data.address,
            user_id: this.data.userId, // 使用获取到的用户唯一标志
            user_info: '', // 替换为实际的用户其他信息
            checkin_time: new Date(),
            created_at: new Date(),
            updated_at: new Date()
          },
          success: function (res) {
            wx.showToast({
              title: '打卡成功',
              icon: 'success',
              duration: 2000
            });
          },
          fail: function (err) {
            console.error('打卡失败', err);
            wx.showToast({
              title: '打卡失败',
              icon: 'none',
              duration: 2000
            });
          }
        });
        this.userLogin()
      },
      fail: (err) => {
        console.error('获取位置信息失败', err);
      }
    });
  },
  // 获取位置名称
  getAddress(latitude, longitude) {
    const ak = 'H1LdMvitgXmoSaIgcCrzqb8K5MzoufTH'; // 替换为你的百度地图API密钥
    const url = `https://api.map.baidu.com/reverse_geocoding/v3/?ak=${ak}&output=json&coordtype=wgs84ll&location=${latitude},${longitude}`;
    wx.request({
      url: url,
      success: (res) => {
        if (res.data && res.data.result && res.data.result.addressComponent) {
          const addressComponent = res.data.result.addressComponent;
          const address = `${addressComponent.province}, ${addressComponent.city}, ${addressComponent.district}, ${addressComponent.street}`;
          this.setData({
            address: address
          });
        } else {
          console.error('获取地名失败', res);
        }
      },
      fail: (err) => {
        console.error('请求失败', err);
      }
    });
  },
  // 用户登录，并获取用户排名
  userLogin() {
    const that = this;
    const app = getApp();
    wx.login({
      success(res) {
        if (res.code) {
          // 调用云函数获取 openId
          app.globalData.cloudInstance.callFunction({
            name: 'invokeLoginGetUserInfo',
            data: {
              code: res.code
            },
            success: function (response) {
              console.log('info:调用函数invokeLoginGetUserInfo', response)
              const openId = response.result.openid;
              that.setData({
                userId: openId
              });
              that.getUserRank(that.data.nearestLocationId, openId).then(rank => {
                that.setData({
                  userRank: rank
                });
              }).catch(err => {
                console.error('获取用户排名失败', err);
              });
            },
            fail: function (err) {
              console.error('获取 openId 失败', err);
              wx.showToast({
                title: '获取 openId 失败',
                icon: 'none',
                duration: 2000
              });
            }
          });
        } else {
          console.log('登录失败！' + res.errMsg);
        }
      }
    });
  },
  // 获取用户排名
  getUserRank(location_id, userId) {
    console.log("info:获取用户排名",location_id, userId)
    const app = getApp();
    const db = app.globalData.cloudInstance.database();
    const _ = db.command;
    // 获取当天的开始时间和结束时间
    const startTime = new Date();
    startTime.setHours(0, 0, 0, 0);
    const endTime = new Date();
    endTime.setHours(23, 59, 59, 999);

    // 返回一个Promise
    return new Promise((resolve, reject) => {
      // 查询符合条件的打卡记录
      db.collection('check_in_record')
        .where({
          location_id: location_id,
          created_at: _.gte(startTime).and(_.lte(endTime))
        })
        .orderBy('created_at', 'asc')
        .get({
          success: (res) => {
            console.log('info:check_in_record集合筛选结果', res)
            // 获取打卡记录列表
            const records = res.data;
            // 查找当前用户的排名
            const userIndex = records.findIndex(record => record._openid === userId);
            if (userIndex !== -1) {
              resolve(userIndex + 1); // 返回用户的排名（从1开始）
            } else {
              resolve('?'); // 用户未在打卡记录中
            }
          },
          fail: (err) => {
            console.error('获取用户排名失败', err);
            reject('?'); // 查询失败
          }
        });
    });
  }

});